-- ==========================================
-- ENTERLIFE EXTENSIONS - CORE WELLBEING TABLES
-- (να τρέξει ΜΕΤΑ το βασικό database/schema.sql)
-- ==========================================

-- 1) Ρυθμίσεις Χρήστη (γλώσσα, theme, ειδοποιήσεις κ.λπ.)
CREATE TABLE IF NOT EXISTS user_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    locale VARCHAR(10) DEFAULT 'el',
    theme ENUM('light','dark','system') DEFAULT 'system',
    timezone VARCHAR(64) DEFAULT 'Europe/Athens',
    daily_screen_time_target INT DEFAULT 480, -- σε λεπτά
    email_notifications TINYINT(1) DEFAULT 1,
    browser_notifications TINYINT(1) DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_settings (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 2) Κατάλογος Ασκήσεων Ματιών
CREATE TABLE IF NOT EXISTS eye_exercises (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    short_description TEXT,
    instructions LONGTEXT NOT NULL,
    difficulty ENUM('easy','medium','hard') DEFAULT 'easy',
    category VARCHAR(100), -- e.g. 'focus', 'relax', 'mobility'
    default_duration_seconds INT DEFAULT 60,
    is_active TINYINT(1) DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 3) Routines Ματιών (συνδυασμός από eye_exercises)
CREATE TABLE IF NOT EXISTS eye_routines (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    description TEXT,
    difficulty ENUM('easy','medium','hard') DEFAULT 'easy',
    total_duration_seconds INT DEFAULT 0,
    is_ai_generated TINYINT(1) DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS eye_routine_exercises (
    id INT AUTO_INCREMENT PRIMARY KEY,
    routine_id INT NOT NULL,
    exercise_id INT NOT NULL,
    sort_order INT NOT NULL DEFAULT 1,
    override_duration_seconds INT DEFAULT NULL,
    repetitions INT DEFAULT 1,
    FOREIGN KEY (routine_id) REFERENCES eye_routines(id) ON DELETE CASCADE,
    FOREIGN KEY (exercise_id) REFERENCES eye_exercises(id) ON DELETE CASCADE,
    UNIQUE KEY unique_routine_exercise (routine_id, exercise_id, sort_order)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 4) Logs Sessions Ματιών
CREATE TABLE IF NOT EXISTS eye_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    routine_id INT NULL,
    started_at DATETIME NOT NULL,
    completed_at DATETIME NULL,
    total_duration_seconds INT DEFAULT 0,
    perceived_strain_before TINYINT NULL, -- 1-5
    perceived_strain_after TINYINT NULL,  -- 1-5
    notes TEXT,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (routine_id) REFERENCES eye_routines(id) ON DELETE SET NULL
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 5) Focus Sessions (Pomodoro / Deep Work)
CREATE TABLE IF NOT EXISTS focus_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    technique ENUM('pomodoro','deep_work','custom') DEFAULT 'pomodoro',
    started_at DATETIME NOT NULL,
    ended_at DATETIME NULL,
    planned_minutes INT DEFAULT 25,
    actual_minutes INT DEFAULT 0,
    interruptions INT DEFAULT 0,
    label VARCHAR(190),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_focus_user_started (user_id, started_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 6) Breathing Sessions
CREATE TABLE IF NOT EXISTS breathing_sessions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    pattern_slug VARCHAR(190), -- e.g. 'box-breathing'
    started_at DATETIME NOT NULL,
    ended_at DATETIME NULL,
    duration_seconds INT DEFAULT 0,
    calm_score TINYINT NULL, -- 1-5 self reported
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_breathing_user_started (user_id, started_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 7) Subscription Plans (Free / Pro κ.λπ.)
CREATE TABLE IF NOT EXISTS subscription_plans (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    name VARCHAR(190) NOT NULL,
    description TEXT,
    price_monthly DECIMAL(10,2) DEFAULT 0.00,
    price_yearly DECIMAL(10,2) DEFAULT 0.00,
    is_active TINYINT(1) DEFAULT 1,
    features_json JSON NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

CREATE TABLE IF NOT EXISTS user_subscriptions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    plan_id INT NOT NULL,
    status ENUM('active','canceled','trial','expired') DEFAULT 'active',
    started_at DATETIME NOT NULL,
    expires_at DATETIME NULL,
    canceled_at DATETIME NULL,
    cancel_reason VARCHAR(255),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (plan_id) REFERENCES subscription_plans(id) ON DELETE RESTRICT,
    INDEX idx_user_subscription (user_id, status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- 8) AI Recommendations (τι προτείνει η AI στο χρήστη)
CREATE TABLE IF NOT EXISTS ai_recommendations (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    rec_type ENUM('routine','education','warning','tip') DEFAULT 'tip',
    payload_json JSON NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_ai_user_created (user_id, created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Seed βασικά subscription plans
INSERT IGNORE INTO subscription_plans (slug, name, description, price_monthly, price_yearly, is_active)
VALUES
('free', 'Free', 'Basic eye exercises and focus tools with limited daily usage.', 0.00, 0.00, 1),
('pro', 'Pro', 'Full access to AI-personalized routines, analytics and unlimited sessions.', 5.99, 59.99, 1);

-- ==========================================
-- ENTERLIFE CONTENT & MARKETING EXTENSIONS
-- ==========================================

-- Κεντρικά settings για site / SEO / tracking / ads / email
CREATE TABLE IF NOT EXISTS site_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(190) NOT NULL UNIQUE,
    setting_value TEXT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Κατηγορίες περιεχομένου (για άρθρα, περιοδικό κ.λπ.)
CREATE TABLE IF NOT EXISTS content_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    name VARCHAR(190) NOT NULL,
    description TEXT NULL,
    content_type ENUM('article','magazine','tool','general') DEFAULT 'article',
    is_active TINYINT(1) DEFAULT 1,
    sort_order INT DEFAULT 1,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Pivot: συσχέτιση άρθρων με κατηγορίες
-- Υποθέτουμε ότι υπάρχει πίνακας articles με id
CREATE TABLE IF NOT EXISTS content_category_article (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    article_id INT NOT NULL,
    FOREIGN KEY (category_id) REFERENCES content_categories(id) ON DELETE CASCADE,
    FOREIGN KEY (article_id) REFERENCES articles(id) ON DELETE CASCADE,
    UNIQUE KEY unique_article_category (category_id, article_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Περιοδικό Enterlife (PDF issues)
CREATE TABLE IF NOT EXISTS magazines (
    id INT AUTO_INCREMENT PRIMARY KEY,
    slug VARCHAR(190) NOT NULL UNIQUE,
    title VARCHAR(190) NOT NULL,
    description TEXT NULL,
    issue_date DATE NOT NULL,
    pdf_path VARCHAR(255) NOT NULL,
    cover_image_path VARCHAR(255) NULL,
    is_public TINYINT(1) DEFAULT 1,
    is_pro_only TINYINT(1) DEFAULT 0,
    download_count INT DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Συσχέτιση περιοδικού με κατηγορίες
CREATE TABLE IF NOT EXISTS content_category_magazine (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    magazine_id INT NOT NULL,
    FOREIGN KEY (category_id) REFERENCES content_categories(id) ON DELETE CASCADE,
    FOREIGN KEY (magazine_id) REFERENCES magazines(id) ON DELETE CASCADE,
    UNIQUE KEY unique_magazine_category (category_id, magazine_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- SEO metadata για άρθρα
CREATE TABLE IF NOT EXISTS article_seo (
    id INT AUTO_INCREMENT PRIMARY KEY,
    article_id INT NOT NULL,
    meta_title VARCHAR(255) NULL,
    meta_description VARCHAR(255) NULL,
    meta_keywords VARCHAR(255) NULL,
    og_title VARCHAR(255) NULL,
    og_description VARCHAR(255) NULL,
    ai_snippet TEXT NULL,
    schema_json JSON NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (article_id) REFERENCES articles(id) ON DELETE CASCADE,
    UNIQUE KEY unique_article_seo (article_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Newsletter subscribers
CREATE TABLE IF NOT EXISTS newsletter_subscribers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(190) NOT NULL UNIQUE,
    name VARCHAR(190) NULL,
    source VARCHAR(190) NULL, -- e.g. 'footer_form', 'magazine_download'
    is_confirmed TINYINT(1) DEFAULT 0,
    confirmation_token VARCHAR(190) NULL,
    unsubscribed_at DATETIME NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- Seed βασικές κατηγορίες περιεχομένου
INSERT IGNORE INTO content_categories (slug, name, description, content_type, sort_order, is_active)
VALUES
('health-eyes', 'Υγεία Ματιών & Οθονών', 'Άρθρα για κούραση ματιών, οθόνες και προστασία.', 'article', 1, 1),
('focus-productivity', 'Focus & Παραγωγικότητα', 'Συγκέντρωση, ρουτίνες, εργασία σε οθόνη.', 'article', 2, 1),
('sleep-wellbeing', 'Ύπνος & Ευεξία', 'Ρυθμός ύπνου, χαλάρωση και ξεκούραση.', 'article', 3, 1),
('health-tech', 'Health & Tech', 'Τεχνολογία που επηρεάζει την υγεία και τη ζωή μας.', 'article', 4, 1),
('digital-life', 'Digital Life', 'Ζωή με οθόνες, social, ισορροπία online/offline.', 'article', 5, 1),
('magazine', 'Enterlife Magazine', 'Τεύχη περιοδικού Enterlife.', 'magazine', 6, 1);

-- Seed βασικά site settings (μπορούν να αλλάξουν από το admin)
INSERT INTO site_settings (setting_key, setting_value) VALUES
('default_meta_title', 'Enterlife - Better Eyes, Better Focus, Better Life'),
('default_meta_description', 'Enterlife είναι η πλατφόρμα ευεξίας για μάτια, συγκέντρωση και ψηφιακή ζωή. Άρθρα, εργαλεία και περιοδικό για μια καλύτερη καθημερινότητα.'),
('default_og_image', '/assets/img/og-default.jpg'),
('tracking_enabled', '0'),
('ga4_measurement_id', ''),
('gtm_container_id', ''),
('adsense_code_head', ''),
('adsense_code_body', ''),
('fb_pixel_id', ''),
('newsletter_sender_email', 'no-reply@enterlife.local'),
('newsletter_sender_name', 'Enterlife'),
('smtp_host', ''),
('smtp_port', ''),
('smtp_username', ''),
('smtp_password', ''),
('smtp_encryption', 'tls')
ON DUPLICATE KEY UPDATE
    setting_value = VALUES(setting_value);
