<?php

class AuthController {
    protected $pdo;
    protected $userModel;

    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->userModel = new UserModel($pdo);
    }

    public function login($email, $password) {
        $email = trim($email);
        $password = (string) $password;

        $user = $this->userModel->findByEmail($email);
        
        if ($user && password_verify($password, $user['password_hash'])) {
            session_regenerate_id(true);
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['user_email'] = $user['email'];
            $_SESSION['user_role'] = $user['role'];
            return true;
        }
        
        return false;
    }

    public function register($email, $password, $name) {
        $email = trim($email);
        $name = trim($name);
        $password = (string) $password;

        if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            return ['error' => 'Invalid email address'];
        }

        if (strlen($password) < 6) {
            return ['error' => 'Password must be at least 6 characters'];
        }

        if ($name === '') {
            return ['error' => 'Name is required'];
        }

        // Check if user exists
        if ($this->userModel->findByEmail($email)) {
            return ['error' => 'Email already exists'];
        }

        $passwordHash = password_hash($password, PASSWORD_DEFAULT);

        // Create user
        $userId = $this->userModel->createUser($email, $passwordHash, $name);
        
        if ($userId) {
            // Auto login after registration
            session_regenerate_id(true);
            $_SESSION['user_id'] = $userId;
            $_SESSION['user_email'] = $email;
            $_SESSION['user_role'] = 'user';
            
            // Initialize stats
            $statsModel = new StatsModel($this->pdo);
            $statsModel->initStats($userId);
            
            return ['success' => true, 'user_id' => $userId];
        }
        
        return ['error' => 'Registration failed'];
    }

    public function logout() {
        session_destroy();
        session_start();
    }
}
