<?php

/**
 * Eye Exercise Controller
 * Handles CRUD operations for eye exercises
 */

class EyeExerciseController {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get all exercises with optional filtering
     * 
     * @param array $filters ['difficulty', 'category', 'is_active', 'search']
     * @return array
     */
    public function getAllExercises($filters = []) {
        $sql = "SELECT * FROM eye_exercises WHERE 1=1";
        $params = [];
        
        if (isset($filters['difficulty'])) {
            $sql .= " AND difficulty = :difficulty";
            $params['difficulty'] = $filters['difficulty'];
        }
        
        if (isset($filters['category'])) {
            $sql .= " AND category = :category";
            $params['category'] = $filters['category'];
        }
        
        if (isset($filters['is_active'])) {
            $sql .= " AND is_active = :is_active";
            $params['is_active'] = $filters['is_active'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND (title LIKE :search OR slug LIKE :search)";
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching exercises: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get exercise by ID
     * 
     * @param int $id
     * @return array|null
     */
    public function getExerciseById($id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM eye_exercises WHERE id = :id");
            $stmt->execute(['id' => $id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ?: null;
        } catch (Exception $e) {
            error_log("Error fetching exercise by ID: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get exercise by slug
     * 
     * @param string $slug
     * @return array|null
     */
    public function getExerciseBySlug($slug) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM eye_exercises WHERE slug = :slug");
            $stmt->execute(['slug' => $slug]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            return $result ?: null;
        } catch (Exception $e) {
            error_log("Error fetching exercise by slug: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Create new exercise
     * 
     * @param array $data Exercise data
     * @return int|false Exercise ID or false on failure
     */
    public function createExercise($data) {
        // Validate data
        $errors = $this->validateExerciseData($data);
        if (!empty($errors)) {
            error_log("Exercise validation failed: " . implode(', ', $errors));
            return false;
        }
        
        // Generate slug if not provided
        if (empty($data['slug'])) {
            $data['slug'] = $this->generateSlug($data['title']);
        } else {
            $data['slug'] = $this->generateSlug($data['slug']);
        }
        
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO eye_exercises (
                    slug, title, short_description, instructions, 
                    difficulty, category, default_duration_seconds, 
                    is_active, created_at, updated_at
                ) VALUES (
                    :slug, :title, :short_description, :instructions,
                    :difficulty, :category, :default_duration_seconds,
                    :is_active, NOW(), NOW()
                )
            ");
            
            $stmt->execute([
                'slug' => $data['slug'],
                'title' => $data['title'],
                'short_description' => $data['short_description'] ?? null,
                'instructions' => $data['instructions'],
                'difficulty' => $data['difficulty'] ?? 'easy',
                'category' => $data['category'] ?? 'general',
                'default_duration_seconds' => $data['default_duration_seconds'],
                'is_active' => $data['is_active'] ?? 1
            ]);
            
            return $this->pdo->lastInsertId();
        } catch (Exception $e) {
            error_log("Error creating exercise: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update existing exercise
     * 
     * @param int $id
     * @param array $data Fields to update
     * @return bool
     */
    public function updateExercise($id, $data) {
        // Validate data
        $errors = $this->validateExerciseData($data, $id);
        if (!empty($errors)) {
            error_log("Exercise validation failed: " . implode(', ', $errors));
            return false;
        }
        
        // Build dynamic update query
        $fields = [];
        $params = ['id' => $id];
        
        $allowedFields = [
            'slug', 'title', 'short_description', 'instructions',
            'difficulty', 'category', 'default_duration_seconds', 'is_active'
        ];
        
        foreach ($allowedFields as $field) {
            if (isset($data[$field])) {
                // Special handling for slug
                if ($field === 'slug') {
                    $data[$field] = $this->generateSlug($data[$field], $id);
                }
                $fields[] = "$field = :$field";
                $params[$field] = $data[$field];
            }
        }
        
        if (empty($fields)) {
            return false; // Nothing to update
        }
        
        // Always update timestamp
        $fields[] = "updated_at = NOW()";
        
        try {
            $sql = "UPDATE eye_exercises SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (Exception $e) {
            error_log("Error updating exercise: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Toggle active status of exercise
     * 
     * @param int $id
     * @return bool
     */
    public function toggleActive($id) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE eye_exercises 
                SET is_active = NOT is_active, updated_at = NOW()
                WHERE id = :id
            ");
            return $stmt->execute(['id' => $id]);
        } catch (Exception $e) {
            error_log("Error toggling exercise active status: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete exercise
     * 
     * @param int $id
     * @return bool
     */
    public function deleteExercise($id) {
        try {
            // Check if exercise is used in any routines
            $stmt = $this->pdo->prepare("
                SELECT COUNT(*) FROM eye_routine_exercises 
                WHERE exercise_id = :id
            ");
            $stmt->execute(['id' => $id]);
            $usageCount = $stmt->fetchColumn();
            
            if ($usageCount > 0) {
                error_log("Cannot delete exercise $id: used in $usageCount routine(s)");
                return false;
            }
            
            // Delete exercise
            $stmt = $this->pdo->prepare("DELETE FROM eye_exercises WHERE id = :id");
            return $stmt->execute(['id' => $id]);
        } catch (Exception $e) {
            error_log("Error deleting exercise: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get all unique categories from existing exercises
     * 
     * @return array
     */
    public function getAllCategories() {
        try {
            $stmt = $this->pdo->query("
                SELECT DISTINCT category 
                FROM eye_exercises 
                WHERE category IS NOT NULL AND category != ''
                ORDER BY category ASC
            ");
            return $stmt->fetchAll(PDO::FETCH_COLUMN);
        } catch (Exception $e) {
            error_log("Error fetching categories: " . $e->getMessage());
            return ['general', 'focus', 'relax', 'mobility', 'strength', 'coordination'];
        }
    }
    
    /**
     * Validate exercise data
     * 
     * @param array $data
     * @param int|null $excludeId Exclude this ID from slug uniqueness check
     * @return array Array of error messages (empty if valid)
     */
    private function validateExerciseData($data, $excludeId = null) {
        $errors = [];
        
        // Title validation
        if (isset($data['title'])) {
            if (empty(trim($data['title']))) {
                $errors[] = "Title is required";
            } elseif (strlen($data['title']) < 3) {
                $errors[] = "Title must be at least 3 characters";
            } elseif (strlen($data['title']) > 190) {
                $errors[] = "Title must not exceed 190 characters";
            }
        }
        
        // Slug validation (if provided)
        if (isset($data['slug']) && !empty($data['slug'])) {
            if (!preg_match('/^[a-z0-9-]+$/', $data['slug'])) {
                $errors[] = "Slug can only contain lowercase letters, numbers, and hyphens";
            }
            if (strlen($data['slug']) > 190) {
                $errors[] = "Slug must not exceed 190 characters";
            }
        }
        
        // Instructions validation
        if (isset($data['instructions'])) {
            if (empty(trim($data['instructions']))) {
                $errors[] = "Instructions are required";
            } elseif (strlen($data['instructions']) < 10) {
                $errors[] = "Instructions must be at least 10 characters";
            }
        }
        
        // Difficulty validation
        if (isset($data['difficulty'])) {
            $validDifficulties = ['easy', 'medium', 'hard'];
            if (!in_array($data['difficulty'], $validDifficulties)) {
                $errors[] = "Difficulty must be one of: " . implode(', ', $validDifficulties);
            }
        }
        
        // Default duration validation
        if (isset($data['default_duration_seconds'])) {
            $duration = (int)$data['default_duration_seconds'];
            if ($duration < 5) {
                $errors[] = "Duration must be at least 5 seconds";
            }
            if ($duration > 3600) {
                $errors[] = "Duration must not exceed 3600 seconds (1 hour)";
            }
        }
        
        // Category validation
        if (isset($data['category'])) {
            if (empty(trim($data['category']))) {
                $errors[] = "Category is required";
            }
            if (strlen($data['category']) > 100) {
                $errors[] = "Category must not exceed 100 characters";
            }
        }
        
        return $errors;
    }
    
    /**
     * Generate unique slug from title
     * 
     * @param string $title
     * @param int|null $excludeId Exclude this ID from uniqueness check
     * @return string
     */
    private function generateSlug($title, $excludeId = null) {
        // Convert to lowercase
        $slug = strtolower($title);
        
        // Replace spaces and special characters with hyphens
        $slug = preg_replace('/[^a-z0-9]+/', '-', $slug);
        
        // Remove consecutive hyphens
        $slug = preg_replace('/-+/', '-', $slug);
        
        // Trim hyphens from start and end
        $slug = trim($slug, '-');
        
        // Ensure uniqueness
        $originalSlug = $slug;
        $counter = 1;
        
        while ($this->slugExists($slug, $excludeId)) {
            $counter++;
            $slug = $originalSlug . '-' . $counter;
        }
        
        return $slug;
    }
    
    /**
     * Check if slug exists in database
     * 
     * @param string $slug
     * @param int|null $excludeId
     * @return bool
     */
    private function slugExists($slug, $excludeId = null) {
        try {
            if ($excludeId) {
                $stmt = $this->pdo->prepare("
                    SELECT COUNT(*) FROM eye_exercises 
                    WHERE slug = :slug AND id != :exclude_id
                ");
                $stmt->execute(['slug' => $slug, 'exclude_id' => $excludeId]);
            } else {
                $stmt = $this->pdo->prepare("
                    SELECT COUNT(*) FROM eye_exercises 
                    WHERE slug = :slug
                ");
                $stmt->execute(['slug' => $slug]);
            }
            
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            error_log("Error checking slug existence: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get statistics about exercises
     * 
     * @return array
     */
    public function getStatistics() {
        try {
            $stats = [];
            
            // Total count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_exercises");
            $stats['total'] = $stmt->fetchColumn();
            
            // Active count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_exercises WHERE is_active = 1");
            $stats['active'] = $stmt->fetchColumn();
            
            // Inactive count
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM eye_exercises WHERE is_active = 0");
            $stats['inactive'] = $stmt->fetchColumn();
            
            // By difficulty
            $stmt = $this->pdo->query("
                SELECT difficulty, COUNT(*) as count 
                FROM eye_exercises 
                GROUP BY difficulty
            ");
            $stats['by_difficulty'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            // By category
            $stmt = $this->pdo->query("
                SELECT category, COUNT(*) as count 
                FROM eye_exercises 
                GROUP BY category 
                ORDER BY count DESC
            ");
            $stats['by_category'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error getting exercise statistics: " . $e->getMessage());
            return [];
        }
    }
}
