<?php

/**
 * Newsletter Controller
 * Handles newsletter subscriber management
 */

class NewsletterController {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get all subscribers with optional filters
     * 
     * @param array $filters ['is_confirmed', 'search']
     * @return array
     */
    public function getAllSubscribers($filters = []) {
        $sql = "SELECT * FROM newsletter_subscribers WHERE 1=1";
        $params = [];
        
        if (isset($filters['is_confirmed'])) {
            $sql .= " AND is_confirmed = :is_confirmed";
            $params['is_confirmed'] = $filters['is_confirmed'];
        }
        
        if (!empty($filters['search'])) {
            $sql .= " AND email LIKE :search";
            $params['search'] = '%' . $filters['search'] . '%';
        }
        
        $sql .= " ORDER BY created_at DESC";
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching subscribers: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get subscriber by email
     * 
     * @param string $email
     * @return array|null
     */
    public function getSubscriberByEmail($email) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM newsletter_subscribers WHERE email = :email");
            $stmt->execute(['email' => $email]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching subscriber: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Add new subscriber
     * 
     * @param string $email
     * @param bool $sendConfirmation
     * @return bool
     */
    public function addSubscriber($email, $sendConfirmation = true) {
        // Check if already exists
        if ($this->getSubscriberByEmail($email)) {
            return false;
        }
        
        try {
            $confirmationToken = bin2hex(random_bytes(32));
            
            $stmt = $this->pdo->prepare("
                INSERT INTO newsletter_subscribers (email, confirmation_token, created_at)
                VALUES (:email, :token, NOW())
            ");
            
            $result = $stmt->execute([
                'email' => $email,
                'token' => $confirmationToken
            ]);
            
            if ($result && $sendConfirmation) {
                $this->sendConfirmationEmail($email, $confirmationToken);
            }
            
            return $result;
        } catch (Exception $e) {
            error_log("Error adding subscriber: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Confirm subscriber
     * 
     * @param string $token
     * @return bool
     */
    public function confirmSubscriber($token) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE newsletter_subscribers 
                SET is_confirmed = 1, confirmed_at = NOW()
                WHERE confirmation_token = :token AND is_confirmed = 0
            ");
            return $stmt->execute(['token' => $token]);
        } catch (Exception $e) {
            error_log("Error confirming subscriber: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Unsubscribe by email or token
     * 
     * @param string $identifier Email or unsubscribe token
     * @return bool
     */
    public function unsubscribe($identifier) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE newsletter_subscribers 
                SET is_unsubscribed = 1, unsubscribed_at = NOW()
                WHERE email = :identifier OR unsubscribe_token = :identifier
            ");
            return $stmt->execute(['identifier' => $identifier]);
        } catch (Exception $e) {
            error_log("Error unsubscribing: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete subscriber
     * 
     * @param int $id
     * @return bool
     */
    public function deleteSubscriber($id) {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM newsletter_subscribers WHERE id = :id");
            return $stmt->execute(['id' => $id]);
        } catch (Exception $e) {
            error_log("Error deleting subscriber: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get subscriber statistics
     * 
     * @return array
     */
    public function getStatistics() {
        try {
            $stats = [];
            
            // Total subscribers
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM newsletter_subscribers WHERE is_unsubscribed = 0");
            $stats['total'] = $stmt->fetchColumn();
            
            // Confirmed subscribers
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM newsletter_subscribers WHERE is_confirmed = 1 AND is_unsubscribed = 0");
            $stats['confirmed'] = $stmt->fetchColumn();
            
            // Pending confirmation
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM newsletter_subscribers WHERE is_confirmed = 0 AND is_unsubscribed = 0");
            $stats['pending'] = $stmt->fetchColumn();
            
            // Unsubscribed
            $stmt = $this->pdo->query("SELECT COUNT(*) FROM newsletter_subscribers WHERE is_unsubscribed = 1");
            $stats['unsubscribed'] = $stmt->fetchColumn();
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error getting statistics: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Export subscribers to CSV
     * 
     * @param bool $confirmedOnly
     * @return string CSV content
     */
    public function exportToCSV($confirmedOnly = true) {
        $filters = ['is_confirmed' => 1];
        if (!$confirmedOnly) {
            unset($filters['is_confirmed']);
        }
        
        $subscribers = $this->getAllSubscribers($filters);
        
        $csv = "Email,Confirmed,Subscribe Date\n";
        foreach ($subscribers as $subscriber) {
            $csv .= sprintf(
                "%s,%s,%s\n",
                $subscriber['email'],
                $subscriber['is_confirmed'] ? 'Yes' : 'No',
                $subscriber['created_at']
            );
        }
        
        return $csv;
    }
    
    /**
     * Send confirmation email (placeholder - implement with actual email service)
     * 
     * @param string $email
     * @param string $token
     * @return bool
     */
    private function sendConfirmationEmail($email, $token) {
        // TODO: Implement with actual email service
        // For now, just log it
        $confirmUrl = base_url("newsletter-confirm.php?token=" . $token);
        error_log("Confirmation email would be sent to {$email}: {$confirmUrl}");
        return true;
    }
}
