<?php

/**
 * Subscription Controller
 * Handles subscription plan and user subscription management
 */

class SubscriptionController {
    private $pdo;
    
    public function __construct($pdo) {
        $this->pdo = $pdo;
    }
    
    /**
     * Get all subscription plans
     * 
     * @param bool $activeOnly
     * @return array
     */
    public function getAllPlans($activeOnly = true) {
        $sql = "SELECT * FROM subscription_plans";
        if ($activeOnly) {
            $sql .= " WHERE is_active = 1";
        }
        $sql .= " ORDER BY price ASC";
        
        try {
            $stmt = $this->pdo->query($sql);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching plans: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get plan by ID
     * 
     * @param int $id
     * @return array|null
     */
    public function getPlanById($id) {
        try {
            $stmt = $this->pdo->prepare("SELECT * FROM subscription_plans WHERE id = :id");
            $stmt->execute(['id' => $id]);
            return $stmt->fetch(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching plan: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Create subscription plan
     * 
     * @param array $data
     * @return int|false Plan ID or false
     */
    public function createPlan($data) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO subscription_plans (
                    plan_name, plan_slug, description, price, currency,
                    billing_cycle, features_json, is_active, created_at
                ) VALUES (
                    :plan_name, :plan_slug, :description, :price, :currency,
                    :billing_cycle, :features_json, :is_active, NOW()
                )
            ");
            
            $stmt->execute([
                'plan_name' => $data['plan_name'],
                'plan_slug' => $data['plan_slug'],
                'description' => $data['description'] ?? null,
                'price' => $data['price'],
                'currency' => $data['currency'] ?? 'USD',
                'billing_cycle' => $data['billing_cycle'] ?? 'monthly',
                'features_json' => $data['features_json'] ?? null,
                'is_active' => $data['is_active'] ?? 1
            ]);
            
            return $this->pdo->lastInsertId();
        } catch (Exception $e) {
            error_log("Error creating plan: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Update subscription plan
     * 
     * @param int $id
     * @param array $data
     * @return bool
     */
    public function updatePlan($id, $data) {
        try {
            $fields = [];
            $params = ['id' => $id];
            
            $allowed = [
                'plan_name', 'plan_slug', 'description', 'price', 'currency',
                'billing_cycle', 'features_json', 'is_active'
            ];
            
            foreach ($allowed as $field) {
                if (isset($data[$field])) {
                    $fields[] = "$field = :$field";
                    $params[$field] = $data[$field];
                }
            }
            
            if (empty($fields)) {
                return false;
            }
            
            $sql = "UPDATE subscription_plans SET " . implode(', ', $fields) . ", updated_at = NOW() WHERE id = :id";
            $stmt = $this->pdo->prepare($sql);
            return $stmt->execute($params);
        } catch (Exception $e) {
            error_log("Error updating plan: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Delete subscription plan
     * 
     * @param int $id
     * @return bool
     */
    public function deletePlan($id) {
        try {
            $stmt = $this->pdo->prepare("DELETE FROM subscription_plans WHERE id = :id");
            return $stmt->execute(['id' => $id]);
        } catch (Exception $e) {
            error_log("Error deleting plan: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get all user subscriptions with user info
     * 
     * @param array $filters ['status' => 'active'|'expired']
     * @return array
     */
    public function getAllUserSubscriptions($filters = []) {
        $sql = "SELECT 
                us.*,
                u.name as user_name,
                u.email as user_email,
                sp.plan_name,
                sp.plan_slug,
                sp.price,
                sp.currency,
                sp.billing_cycle
            FROM user_subscriptions us
            JOIN users u ON us.user_id = u.id
            JOIN subscription_plans sp ON us.plan_id = sp.id
            WHERE 1=1";
        
        $params = [];
        
        if (isset($filters['status'])) {
            if ($filters['status'] === 'active') {
                $sql .= " AND us.start_date <= NOW() AND (us.end_date IS NULL OR us.end_date >= NOW())";
            } elseif ($filters['status'] === 'expired') {
                $sql .= " AND us.end_date < NOW()";
            }
        }
        
        if (isset($filters['plan_id'])) {
            $sql .= " AND us.plan_id = :plan_id";
            $params['plan_id'] = $filters['plan_id'];
        }
        
        $sql .= " ORDER BY us.created_at DESC";
        
        try {
            $stmt = $this->pdo->prepare($sql);
            $stmt->execute($params);
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (Exception $e) {
            error_log("Error fetching user subscriptions: " . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Grant subscription to user (manual admin action)
     * 
     * @param int $userId
     * @param int $planId
     * @param string $startDate
     * @param string $endDate
     * @return bool
     */
    public function grantSubscription($userId, $planId, $startDate = null, $endDate = null) {
        $startDate = $startDate ?? date('Y-m-d');
        
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO user_subscriptions (user_id, plan_id, start_date, end_date, created_at)
                VALUES (:user_id, :plan_id, :start_date, :end_date, NOW())
            ");
            
            return $stmt->execute([
                'user_id' => $userId,
                'plan_id' => $planId,
                'start_date' => $startDate,
                'end_date' => $endDate
            ]);
        } catch (Exception $e) {
            error_log("Error granting subscription: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Revoke/cancel user subscription
     * 
     * @param int $subscriptionId
     * @return bool
     */
    public function revokeSubscription($subscriptionId) {
        try {
            $stmt = $this->pdo->prepare("
                UPDATE user_subscriptions 
                SET end_date = NOW(), updated_at = NOW()
                WHERE id = :id
            ");
            return $stmt->execute(['id' => $subscriptionId]);
        } catch (Exception $e) {
            error_log("Error revoking subscription: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get subscription statistics
     * 
     * @return array
     */
    public function getStatistics() {
        try {
            $stats = [];
            
            // Active subscriptions
            $stmt = $this->pdo->query("
                SELECT COUNT(*) FROM user_subscriptions 
                WHERE start_date <= NOW() 
                AND (end_date IS NULL OR end_date >= NOW())
            ");
            $stats['active'] = $stmt->fetchColumn();
            
            // Expired subscriptions
            $stmt = $this->pdo->query("
                SELECT COUNT(*) FROM user_subscriptions 
                WHERE end_date < NOW()
            ");
            $stats['expired'] = $stmt->fetchColumn();
            
            // Subscriptions by plan
            $stmt = $this->pdo->query("
                SELECT sp.plan_name, COUNT(*) as count
                FROM user_subscriptions us
                JOIN subscription_plans sp ON us.plan_id = sp.id
                WHERE us.start_date <= NOW() 
                AND (us.end_date IS NULL OR us.end_date >= NOW())
                GROUP BY sp.id, sp.plan_name
            ");
            $stats['by_plan'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            return $stats;
        } catch (Exception $e) {
            error_log("Error getting subscription statistics: " . $e->getMessage());
            return [];
        }
    }
}
