<?php

class ChallengeModel {
    protected $pdo;

    public function __construct($pdo) {
        $this->pdo = $pdo;
    }

    public function getDailyChallenge($userId) {
        $today = date('Y-m-d');
        
        // Check if user already has a challenge assigned for today
        $stmt = $this->pdo->prepare("
            SELECT c.*, uc.completed, uc.id as assignment_id 
            FROM challenges c 
            JOIN user_challenges uc ON c.id = uc.challenge_id 
            WHERE uc.user_id = :user_id AND uc.date_assigned = :today
            LIMIT 1
        ");
        $stmt->execute(['user_id' => $userId, 'today' => $today]);
        $challenge = $stmt->fetch();

        if ($challenge) {
            return $challenge;
        }

        // If not, assign a random one
        return $this->assignRandomChallenge($userId, $today);
    }

    public function assignRandomChallenge($userId, $today) {
        // Get a random challenge
        $stmt = $this->pdo->query("SELECT * FROM challenges ORDER BY RAND() LIMIT 1");
        $challenge = $stmt->fetch();

        if (!$challenge) {
            // Create default challenges if none exist
            $this->seedChallenges();
            $stmt = $this->pdo->query("SELECT * FROM challenges ORDER BY RAND() LIMIT 1");
            $challenge = $stmt->fetch();
        }

        // Assign it
        $stmt = $this->pdo->prepare("INSERT INTO user_challenges (user_id, challenge_id, date_assigned) VALUES (:user_id, :challenge_id, :today)");
        $stmt->execute(['user_id' => $userId, 'challenge_id' => $challenge['id'], 'today' => $today]);
        
        return array_merge($challenge, ['completed' => 0, 'assignment_id' => $this->pdo->lastInsertId()]);
    }

    public function completeChallenge($userId, $assignmentId) {
        $stmt = $this->pdo->prepare("UPDATE user_challenges SET completed = 1, completed_at = NOW() WHERE id = :id AND user_id = :user_id");
        return $stmt->execute(['id' => $assignmentId, 'user_id' => $userId]);
    }

    public function getStreak($userId) {
        // Simple streak calculation: consecutive days with completed challenges ending yesterday or today
        // This is a simplified version. For a robust streak, we'd need more complex SQL or logic.
        // Let's just count total completed for now as a "score" or implement a basic check.
        
        $stmt = $this->pdo->prepare("SELECT COUNT(*) FROM user_challenges WHERE user_id = :user_id AND completed = 1");
        $stmt->execute(['user_id' => $userId]);
        return $stmt->fetchColumn();
    }

    private function seedChallenges() {
        $challenges = [
            ['title' => 'Drink Water', 'description' => 'Drink 8 glasses of water today.'],
            ['title' => 'Take a Walk', 'description' => 'Go for a 15-minute walk outside.'],
            ['title' => 'Read a Book', 'description' => 'Read 10 pages of a book.'],
            ['title' => 'No Sugar', 'description' => 'Avoid sugary drinks and snacks today.'],
            ['title' => 'Meditate', 'description' => 'Do a 5-minute breathing exercise.'],
            ['title' => 'Digital Detox', 'description' => 'No social media for 1 hour.'],
            ['title' => 'Gratitude', 'description' => 'Write down 3 things you are grateful for.'],
        ];

        $stmt = $this->pdo->prepare("INSERT INTO challenges (title, description) VALUES (:title, :description)");
        foreach ($challenges as $c) {
            $stmt->execute($c);
        }
    }
}
