<?php

/**
 * Subscription Management Functions
 * Handles PRO/FREE tier logic and user subscription checks
 */

/**
 * Get user's current subscription level
 * 
 * @param int $userId
 * @return string|null 'free', 'pro', 'premium', etc. or null if no subscription
 */
function getUserSubscriptionLevel($userId) {
    global $pdo;
    
    if (!$userId) {
        return 'free';
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT sp.plan_slug, us.end_date
            FROM user_subscriptions us
            JOIN subscription_plans sp ON us.plan_id = sp.id
            WHERE us.user_id = :user_id
            AND us.start_date <= NOW()
            AND (us.end_date IS NULL OR us.end_date >= NOW())
            ORDER BY sp.price DESC
            LIMIT 1
        ");
        $stmt->execute(['user_id' => $userId]);
        $subscription = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($subscription) {
            return $subscription['plan_slug']; // 'pro', 'premium', etc.
        }
        
        return 'free';
    } catch (Exception $e) {
        error_log("Error getting subscription level: " . $e->getMessage());
        return 'free';
    }
}

/**
 * Check if user has PRO subscription
 * 
 * @param int $userId
 * @return bool
 */
function isPro($userId) {
    $level = getUserSubscriptionLevel($userId);
    return in_array($level, ['pro', 'premium', 'enterprise']);
}

/**
 * Check if user has active subscription (any paid plan)
 * 
 * @param int $userId
 * @return bool
 */
function hasActiveSubscription($userId) {
    $level = getUserSubscriptionLevel($userId);
    return $level !== 'free';
}

/**
 * Require PRO subscription or redirect
 * 
 * @param string $redirectUrl URL to redirect to if not PRO (default: upgrade.php)
 */
function requirePro($redirectUrl = null) {
    if (!isLoggedIn()) {
        redirect(base_url('login.php?redirect=' . urlencode($_SERVER['REQUEST_URI'])));
    }
    
    if (!isPro($_SESSION['user_id'])) {
        $url = $redirectUrl ?? base_url('upgrade.php?reason=pro_required');
        redirect($url);
    }
}

/**
 * Get user subscription details
 * 
 * @param int $userId
 * @return array|null
 */
function getUserSubscription($userId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                us.*,
                sp.plan_name,
                sp.plan_slug,
                sp.price,
                sp.currency,
                sp.billing_cycle
            FROM user_subscriptions us
            JOIN subscription_plans sp ON us.plan_id = sp.id
            WHERE us.user_id = :user_id
            AND us.start_date <= NOW()
            AND (us.end_date IS NULL OR us.end_date >= NOW())
            ORDER BY us.start_date DESC
            LIMIT 1
        ");
        $stmt->execute(['user_id' => $userId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Error getting user subscription: " . $e->getMessage());
        return null;
    }
}

/**
 * Check if content requires PRO access
 * 
 * @param array $content Content array with is_pro_only field
 * @return bool
 */
function isProOnly($content) {
    return isset($content['is_pro_only']) && $content['is_pro_only'] == 1;
}

/**
 * Gate content based on PRO requirement
 * Returns true if user can access, false otherwise
 * 
 * @param array $content
 * @param int $userId
 * @return bool
 */
function canAccessContent($content, $userId) {
    if (!isProOnly($content)) {
        return true; // Free content, everyone can access
    }
    
    return isPro($userId);
}

/**
 * Get all subscription plans
 * 
 * @return array
 */
function getAllSubscriptionPlans() {
    global $pdo;
    
    try {
        $stmt = $pdo->query("
            SELECT * FROM subscription_plans 
            WHERE is_active = 1 
            ORDER BY price ASC
        ");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        error_log("Error fetching subscription plans: " . $e->getMessage());
        return [];
    }
}

/**
 * Create or update user subscription
 * 
 * @param int $userId
 * @param int $planId
 * @param string $startDate
 * @param string|null $endDate
 * @return bool
 */
function createUserSubscription($userId, $planId, $startDate, $endDate = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO user_subscriptions (user_id, plan_id, start_date, end_date)
            VALUES (:user_id, :plan_id, :start_date, :end_date)
        ");
        return $stmt->execute([
            'user_id' => $userId,
            'plan_id' => $planId,
            'start_date' => $startDate,
            'end_date' => $endDate
        ]);
    } catch (Exception $e) {
        error_log("Error creating subscription: " . $e->getMessage());
        return false;
    }
}

/**
 * Cancel user subscription
 * 
 * @param int $subscriptionId
 * @return bool
 */
function cancelUserSubscription($subscriptionId) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE user_subscriptions 
            SET end_date = NOW() 
            WHERE id = :id
        ");
        return $stmt->execute(['id' => $subscriptionId]);
    } catch (Exception $e) {
        error_log("Error canceling subscription: " . $e->getMessage());
        return false;
    }
}
