<?php

/**
 * Minimal Supabase helper for management and project REST calls.
 * Assumes credentials are supplied via env-backed constants in config.php.
 */
class SupabaseClient
{
    private string $projectUrl;
    private string $apiKey;
    private string $accessToken;

    public function __construct(?string $projectUrl = null, ?string $apiKey = null, ?string $accessToken = null)
    {
        $this->projectUrl = rtrim($projectUrl ?? SUPABASE_URL ?? '', '/');
        $this->apiKey = $apiKey ?? (SUPABASE_SERVICE_ROLE_KEY ?: SUPABASE_ANON_KEY);
        $this->accessToken = $accessToken ?? SUPABASE_ACCESS_TOKEN;
    }

    public function hasProjectConfig(): bool
    {
        return $this->projectUrl !== '' && $this->apiKey !== '';
    }

    public function hasAccessToken(): bool
    {
        return $this->accessToken !== '';
    }

    /**
     * Calls the Supabase management API to list projects to verify the PAT works.
     */
    public function listProjects(): array
    {
        if (!$this->hasAccessToken()) {
            return [
                'ok' => false,
                'status' => null,
                'error' => 'Missing SUPABASE_ACCESS_TOKEN for management API.',
            ];
        }

        return $this->managementRequest('GET', '/v1/projects');
    }

    /**
     * Pings the project's auth service to confirm URL + API key are valid.
     */
    public function projectHealth(): array
    {
        if (!$this->hasProjectConfig()) {
            return [
                'ok' => false,
                'status' => null,
                'error' => 'Missing SUPABASE_URL or key.',
            ];
        }

        return $this->projectRequest('GET', '/auth/v1/health');
    }

    /**
     * Generic request to the project's REST endpoints.
     */
    public function projectRequest(string $method, string $path, array $query = [], $body = null): array
    {
        $url = $this->buildUrl($this->projectUrl . $path, $query);

        $headers = [
            'apikey: ' . $this->apiKey,
            'Authorization: Bearer ' . $this->apiKey,
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        return $this->httpRequest($method, $url, $headers, $body);
    }

    /**
     * Request against the management API (uses personal access token).
     */
    public function managementRequest(string $method, string $path, array $query = [], $body = null): array
    {
        $url = $this->buildUrl('https://api.supabase.com' . $path, $query);

        $headers = [
            'Authorization: Bearer ' . $this->accessToken,
            'Content-Type: application/json',
            'Accept: application/json',
        ];

        return $this->httpRequest($method, $url, $headers, $body);
    }

    private function buildUrl(string $url, array $query = []): string
    {
        if (!empty($query)) {
            $url .= (str_contains($url, '?') ? '&' : '?') . http_build_query($query);
        }
        return $url;
    }

    private function httpRequest(string $method, string $url, array $headers = [], $body = null): array
    {
        $ch = curl_init();

        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => strtoupper($method),
            CURLOPT_HTTPHEADER => $headers,
        ];

        if ($body !== null) {
            $payload = is_string($body) ? $body : json_encode($body);
            $options[CURLOPT_POSTFIELDS] = $payload;
        }

        curl_setopt_array($ch, $options);

        $response = curl_exec($ch);
        $error = curl_error($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($error) {
            return [
                'ok' => false,
                'status' => $status ?: null,
                'error' => $error,
                'body' => null,
            ];
        }

        $decoded = null;
        if ($response !== false && $response !== '') {
            $decoded = json_decode($response, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $decoded = $response;
            }
        }

        $ok = $status >= 200 && $status < 300;

        return [
            'ok' => $ok,
            'status' => $status,
            'body' => $decoded,
            'error' => $ok ? null : ($decoded['message'] ?? $decoded ?? 'Request failed'),
        ];
    }
}
