<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$articleModel = new ArticleModel(getPDO());
$action = $_GET['action'] ?? 'list';
$id = isset($_GET['id']) ? (int) $_GET['id'] : null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    if ($action === 'create' || $action === 'edit') {
        $title = trim($_POST['title'] ?? '');
        $slug = strtolower(trim($_POST['slug'] ?? ''));
        $body = trim($_POST['body'] ?? '');
        $excerpt = trim($_POST['excerpt'] ?? '');
        $category = trim($_POST['category'] ?? '');
        $published = isset($_POST['published']) ? 1 : 0;

        // Basic validation to prevent bad data / XSS
        $redirectTarget = $action === 'edit' ? "articles.php?action=edit&id={$id}" : "articles.php?action=create";

        if ($title === '' || strlen($title) > 190) {
            flash('error', 'Title is required and must be under 190 characters.');
            redirect($redirectTarget);
        }

        if ($slug === '' || strlen($slug) > 190 || !preg_match('/^[a-z0-9-]+$/', $slug)) {
            flash('error', 'Slug must be lowercase letters, numbers, and hyphens only (max 190 chars).');
            redirect($redirectTarget);
        }

        if ($body === '') {
            flash('error', 'Body is required.');
            redirect($redirectTarget);
        }

        if (strlen($category) > 100) {
            flash('error', 'Category must be under 100 characters.');
            redirect($redirectTarget);
        }

        if (strlen($excerpt) > 500) {
            flash('error', 'Excerpt must be under 500 characters.');
            redirect($redirectTarget);
        }
        
        if ($action === 'create') {
            $articleModel->createArticle($title, $slug, $body, $excerpt, $category, $published);
            flash('success', 'Article created.');
        } else {
            $articleModel->updateArticle($id, $title, $slug, $body, $excerpt, $category, $published);
            flash('success', 'Article updated.');
        }
        redirect('articles.php');
    } elseif ($action === 'delete') {
        if ($id) {
            $articleModel->deleteArticle($id);
            flash('success', 'Article deleted.');
        } else {
            flash('error', 'Invalid article id.');
        }
        redirect('articles.php');
    }
}

include __DIR__ . '/../../templates/layout/header.php';
?>

<h2>Manage Articles</h2>
<a href="articles.php?action=create" class="btn btn-primary">New Article</a>
<hr>

<?php if ($action === 'list'): ?>
    <?php $articles = $articleModel->getAllArticles(); ?>
    <table border="1" cellpadding="10" style="width:100%; border-collapse: collapse;">
        <thead>
            <tr>
                <th>ID</th>
                <th>Title</th>
                <th>Status</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($articles as $article): ?>
                <tr>
                    <td><?= $article['id'] ?></td>
                    <td><?= e($article['title']) ?></td>
                    <td><?= $article['published'] ? 'Published' : 'Draft' ?></td>
                    <td>
                        <a href="articles.php?action=edit&id=<?= $article['id'] ?>">Edit</a>
                        <form action="articles.php?action=delete&id=<?= $article['id'] ?>" method="POST" style="display:inline;" onsubmit="return confirm('Are you sure?');">
                            <?= csrf_field() ?>
                            <button type="submit">Delete</button>
                        </form>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

<?php elseif ($action === 'create' || $action === 'edit'): ?>
    <?php
    $article = null;
    if ($action === 'edit') {
        $article = $articleModel->getArticleById($id);
        if (!$article) {
            flash('error', 'Article not found.');
            redirect('articles.php');
        }
    }
    ?>
    <form method="POST">
        <?= csrf_field() ?>
        <div class="form-group">
            <label>Title</label>
            <input type="text" name="title" class="form-control" value="<?= e($article['title'] ?? '') ?>" required>
        </div>
        <div class="form-group">
            <label>Slug</label>
            <input type="text" name="slug" class="form-control" value="<?= e($article['slug'] ?? '') ?>" required>
        </div>
        <div class="form-group">
            <label>Category</label>
            <input type="text" name="category" class="form-control" value="<?= e($article['category'] ?? '') ?>">
        </div>
        <div class="form-group">
            <label>Excerpt</label>
            <textarea name="excerpt" class="form-control"><?= e($article['excerpt'] ?? '') ?></textarea>
        </div>
        <div class="form-group">
            <label>Body</label>
            <textarea name="body" class="form-control" rows="10" required><?= e($article['body'] ?? '') ?></textarea>
        </div>
        <div class="form-group">
            <label>
                <input type="checkbox" name="published" <?= (!empty($article['published'])) ? 'checked' : '' ?>> Published
            </label>
        </div>
        <button type="submit" class="btn btn-success">Save</button>
        <a href="articles.php" class="btn btn-secondary">Cancel</a>
    </form>
<?php endif; ?>

<?php include __DIR__ . '/../templates/layout/footer.php'; ?>
