<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$exerciseController = new EyeExerciseController(getPDO());

$editMode = false;
$exercise = null;

// Load exercise for editing
if (isset($_GET['id'])) {
    $editMode = true;
    $exercise = $exerciseController->getExerciseById($_GET['id']);
    if (!$exercise) {
        flash('flash_message', 'Exercise not found.', 'danger');
        redirect(base_url('admin/eye-exercises.php'));
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    // Build data array
    $data = [
        'title' => $_POST['title'] ?? '',
        'slug' => $_POST['slug'] ?? '',
        'short_description' => $_POST['short_description'] ?? '',
        'instructions' => $_POST['instructions'] ?? '',
        'category' => $_POST['category'] ?? '',
        'difficulty' => $_POST['difficulty'] ?? 'easy',
        'default_duration_seconds' => (int)($_POST['default_duration_seconds'] ?? 60),
        'is_active' => isset($_POST['is_active']) ? 1 : 0
    ];
    
    if ($editMode) {
        $exerciseId = $_POST['exercise_id'] ?? null;
        if ($exerciseController->updateExercise($exerciseId, $data)) {
            flash('flash_message', 'Exercise updated successfully!', 'success');
            redirect(base_url('admin/eye-exercises.php'));
        } else {
            flash('flash_message', 'Error updating exercise. Check all required fields.', 'danger');
        }
    } else {
        $exerciseId = $exerciseController->createExercise($data);
        if ($exerciseId) {
            flash('flash_message', 'Exercise created successfully!', 'success');
            redirect(base_url('admin/eye-exercises.php'));
        } else {
            flash('flash_message', 'Error creating exercise. Check all required fields.', 'danger');
        }
    }
}

$categories = $exerciseController->getAllCategories();

$pageTitle = ($editMode ? 'Edit' : 'Create New') . ' Eye Exercise - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 900px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .admin-nav a.active { background: #28a745; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; }
        .form-group label .required { color: #dc3545; }
        .form-group input, .form-group textarea, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-size: 14px; font-family: inherit; }
        .form-group textarea { min-height: 100px; resize: vertical; }
        .form-group small { display: block; color: #666; margin-top: 5px; font-size: 13px; }
        .form-group .hint { color: #007bff; font-style: italic; }
        .checkbox-group { margin-top: 10px; }
        .checkbox-group label { display: inline-flex; align-items: center; font-weight: normal; }
        .checkbox-group input { width: auto; margin-right: 8px; }
        .btn { padding: 12px 24px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-secondary { background: #6c757d; }
        .btn-secondary:hover { background: #5a6268; }
        .btn-danger { background: #dc3545; }
        .btn-danger:hover { background: #c82333; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .duration-helper { display: flex; gap: 10px; align-items: center; }
        .duration-helper input { width: 80px; }
        .char-counter { float: right; font-size: 12px; color: #999; }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1><?= $editMode ? 'Edit' : 'Create New' ?> Eye Exercise</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <form method="POST">
            <?= csrf_field() ?>
            <?php if ($editMode): ?>
                <input type="hidden" name="exercise_id" value="<?= e($exercise['id']) ?>">
            <?php endif; ?>
            
            <div class="form-group">
                <label for="title">
                    Title <span class="required">*</span>
                    <span class="char-counter" id="title-counter">0/190</span>
                </label>
                <input type="text" id="title" name="title" required maxlength="190"
                       value="<?= $editMode ? e($exercise['title']) : '' ?>"
                       oninput="updateCharCounter('title', 190)">
                <small>User-friendly name for the exercise</small>
            </div>
            
            <div class="form-group">
                <label for="slug">
                    Slug <span class="required">*</span>
                    <span class="char-counter" id="slug-counter">0/190</span>
                </label>
                <input type="text" id="slug" name="slug" maxlength="190"
                       value="<?= $editMode ? e($exercise['slug']) : '' ?>"
                       oninput="updateCharCounter('slug', 190)">
                <small class="hint">URL-friendly identifier (auto-generated from title if left empty). Use lowercase, numbers, and hyphens only.</small>
            </div>
            
            <div class="form-group">
                <label for="short_description">
                    Short Description
                    <span class="char-counter" id="desc-counter">0/500</span>
                </label>
                <textarea id="short_description" name="short_description" maxlength="500" rows="3"
                          oninput="updateCharCounter('short_description', 500)"><?= $editMode ? e($exercise['short_description']) : '' ?></textarea>
                <small>Brief summary displayed in listings (recommended ~150 characters)</small>
            </div>
            
            <div class="form-group">
                <label for="instructions">Instructions <span class="required">*</span></label>
                <textarea id="instructions" name="instructions" required rows="8"><?= $editMode ? e($exercise['instructions']) : '' ?></textarea>
                <small>Detailed step-by-step instructions. You can use basic HTML if needed.</small>
            </div>
            
            <div class="form-group">
                <label for="category">Category <span class="required">*</span></label>
                <input type="text" id="category" name="category" required list="category-list"
                       value="<?= $editMode ? e($exercise['category']) : '' ?>">
                <datalist id="category-list">
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?= e($cat) ?>">
                    <?php endforeach; ?>
                    <option value="focus">
                    <option value="relax">
                    <option value="mobility">
                    <option value="strength">
                    <option value="coordination">
                </datalist>
                <small>Select existing category or type a new one (e.g., focus, relax, mobility)</small>
            </div>
            
            <div class="form-group">
                <label for="difficulty">Difficulty <span class="required">*</span></label>
                <select id="difficulty" name="difficulty" required>
                    <option value="easy" <?= ($editMode && $exercise['difficulty'] === 'easy') ? 'selected' : '' ?>>Easy</option>
                    <option value="medium" <?= ($editMode && $exercise['difficulty'] === 'medium') ? 'selected' : '' ?>>Medium</option>
                    <option value="hard" <?= ($editMode && $exercise['difficulty'] === 'hard') ? 'selected' : '' ?>>Hard</option>
                </select>
            </div>
            
            <div class="form-group">
                <label for="default_duration_seconds">
                    Default Duration <span class="required">*</span>
                </label>
                <div class="duration-helper">
                    <input type="number" id="duration_minutes" min="0" max="60" 
                           value="<?= $editMode ? floor($exercise['default_duration_seconds'] / 60) : 1 ?>"
                           oninput="updateDurationSeconds()"> minutes
                    <input type="number" id="duration_seconds_part" min="0" max="59" 
                           value="<?= $editMode ? ($exercise['default_duration_seconds'] % 60) : 0 ?>"
                           oninput="updateDurationSeconds()"> seconds
                </div>
                <input type="hidden" id="default_duration_seconds" name="default_duration_seconds" 
                       value="<?= $editMode ? e($exercise['default_duration_seconds']) : 60 ?>">
                <small>Total: <strong id="total-duration">1m 0s</strong> (must be between 5 seconds and 1 hour)</small>
            </div>
            
            <div class="checkbox-group">
                <label>
                    <input type="checkbox" name="is_active" value="1" 
                           <?= ($editMode && $exercise['is_active']) || !$editMode ? 'checked' : '' ?>>
                    Active (visible to users)
                </label>
            </div>
            
            <div style="margin-top: 30px; padding-top: 20px; border-top: 2px solid #ddd;">
                <button type="submit" class="btn"><?= $editMode ? 'Update' : 'Create' ?> Exercise</button>
                <a href="<?= base_url('admin/eye-exercises.php') ?>" class="btn btn-secondary">Cancel</a>
                
                <?php if ($editMode): ?>
                    <form method="POST" action="<?= base_url('admin/eye-exercises.php') ?>" style="display: inline; float: right;"
                          onsubmit="return confirm('Delete this exercise? This cannot be undone.');">
                        <?= csrf_field() ?>
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="id" value="<?= e($exercise['id']) ?>">
                        <button type="submit" class="btn btn-danger">Delete Exercise</button>
                    </form>
                <?php endif; ?>
            </div>
        </form>
    </div>
    
    <script>
        // Auto-generate slug from title
        document.getElementById('title').addEventListener('input', function(e) {
            const slugField = document.getElementById('slug');
            if (!slugField.dataset.manuallyEdited) {
                const slug = e.target.value
                    .toLowerCase()
                    .replace(/[^a-z0-9]+/g, '-')
                    .replace(/^-+|-+$/g, '');
                slugField.value = slug;
                updateCharCounter('slug', 190);
            }
        });
        
        // Mark slug as manually edited if user types in it
        document.getElementById('slug').addEventListener('input', function() {
            this.dataset.manuallyEdited = 'true';
        });
        
        // Character counter
        function updateCharCounter(fieldId, max) {
            const field = document.getElementById(fieldId);
            const counter = document.getElementById(fieldId + '-counter');
            if (counter && field) {
                const current = field.value.length;
                counter.textContent = current + '/' + max;
                counter.style.color = current > max * 0.9 ? '#dc3545' : '#999';
            }
        }
        
        // Duration calculator
        function updateDurationSeconds() {
            const mins = parseInt(document.getElementById('duration_minutes').value) || 0;
            const secs = parseInt(document.getElementById('duration_seconds_part').value) || 0;
            const total = (mins * 60) + secs;
            
            document.getElementById('default_duration_seconds').value = total;
            document.getElementById('total-duration').textContent = mins + 'm ' + secs + 's';
        }
        
        // Initialize counters on load
        document.addEventListener('DOMContentLoaded', function() {
            updateCharCounter('title', 190);
            updateCharCounter('slug', 190);
            updateCharCounter('short_description', 500);
            updateDurationSeconds();
        });
    </script>
</body>
</html>
