<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$exerciseController = new EyeExerciseController(getPDO());

// Handle quick actions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    verify_csrf();
    
    $action = $_POST['action'];
    $id = $_POST['id'] ?? null;
    
    if ($action === 'toggle_active' && $id) {
        if ($exerciseController->toggleActive($id)) {
            flash('flash_message', 'Exercise status updated!', 'success');
        } else {
            flash('flash_message', 'Error updating status.', 'danger');
        }
    } elseif ($action === 'delete' && $id) {
        if ($exerciseController->deleteExercise($id)) {
            flash('flash_message', 'Exercise deleted successfully!', 'success');
        } else {
            flash('flash_message', 'Cannot delete exercise (may be used in routines).', 'danger');
        }
    }
    
    redirect(base_url('admin/eye-exercises.php'));
}

// Get filters
$filters = [];
if (!empty($_GET['difficulty'])) {
    $filters['difficulty'] = $_GET['difficulty'];
}
if (isset($_GET['is_active']) && $_GET['is_active'] !== '') {
    $filters['is_active'] = $_GET['is_active'];
}
if (!empty($_GET['category'])) {
    $filters['category'] = $_GET['category'];
}
if (!empty($_GET['search'])) {
    $filters['search'] = $_GET['search'];
}

$exercises = $exerciseController->getAllExercises($filters);
$categories = $exerciseController->getAllCategories();
$stats = $exerciseController->getStatistics();

$pageTitle = 'Eye Exercises - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 1400px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .admin-nav a.active { background: #28a745; }
        .stats-grid { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 30px; }
        .stat-card { background: #f8f9fa; border: 1px solid #ddd; border-radius: 8px; padding: 15px; text-align: center; }
        .stat-card h3 { margin: 0; font-size: 28px; color: #007bff; }
        .stat-card p { margin: 5px 0 0; color: #666; font-size: 14px; }
        .filters { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; display: flex; gap: 10px; flex-wrap: wrap; align-items: center; }
        .filters select, .filters input { padding: 8px; border: 1px solid #ddd; border-radius: 5px; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-danger { background: #dc3545; }
        .btn-danger:hover { background: #c82333; }
        .btn-sm { padding: 5px 10px; font-size: 12px; }
        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
        th, td { padding: 12px; text-align: left; border-bottom: 1px solid #ddd; }
        th { background: #f8f9fa; font-weight: bold; }
        tr:hover { background: #f8f9fa; }
        .badge { padding: 4px 8px; border-radius: 3px; font-size: 11px; font-weight: bold; text-transform: uppercase; }
        .badge-easy { background: #28a745; color: white; }
        .badge-medium { background: #ffc107; color: #333; }
        .badge-hard { background: #dc3545; color: white; }
        .badge-category { background: #6c757d; color: white; }
        .badge-active { background: #28a745; color: white; }
        .badge-inactive { background: #dc3545; color: white; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .toggle-switch { position: relative; display: inline-block; width: 50px; height: 24px; }
        .toggle-switch input { opacity: 0; width: 0; height: 0; }
        .slider { position: absolute; cursor: pointer; top: 0; left: 0; right: 0; bottom: 0; background-color: #ccc; transition: .4s; border-radius: 24px; }
        .slider:before { position: absolute; content: ""; height: 16px; width: 16px; left: 4px; bottom: 4px; background-color: white; transition: .4s; border-radius: 50%; }
        input:checked + .slider { background-color: #28a745; }
        input:checked + .slider:before { transform: translateX(26px); }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1>Admin Panel - Eye Exercises</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <div class="stats-grid">
            <div class="stat-card">
                <h3><?= e($stats['total']) ?></h3>
                <p>Total Exercises</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['active']) ?></h3>
                <p>Active</p>
            </div>
            <div class="stat-card">
                <h3><?= e($stats['inactive']) ?></h3>
                <p>Inactive</p>
            </div>
        </div>
        
        <div style="margin-bottom: 20px;">
            <a href="<?= base_url('admin/eye-exercises-edit.php') ?>" class="btn">+ Create New Exercise</a>
        </div>
        
        <div class="filters">
            <form method="GET" style="display: flex; gap: 10px; flex-wrap: wrap; width: 100%;">
                <select name="difficulty" onchange="this.form.submit()">
                    <option value="">All Difficulties</option>
                    <option value="easy" <?= ($_GET['difficulty'] ?? '') === 'easy' ? 'selected' : '' ?>>Easy</option>
                    <option value="medium" <?= ($_GET['difficulty'] ?? '') === 'medium' ? 'selected' : '' ?>>Medium</option>
                    <option value="hard" <?= ($_GET['difficulty'] ?? '') === 'hard' ? 'selected' : '' ?>>Hard</option>
                </select>
                
                <select name="category" onchange="this.form.submit()">
                    <option value="">All Categories</option>
                    <?php foreach ($categories as $cat): ?>
                        <option value="<?= e($cat) ?>" <?= ($_GET['category'] ?? '') === $cat ? 'selected' : '' ?>>
                            <?= e(ucfirst($cat)) ?>
                        </option>
                    <?php endforeach; ?>
                </select>
                
                <select name="is_active" onchange="this.form.submit()">
                    <option value="">All Status</option>
                    <option value="1" <?= ($_GET['is_active'] ?? '') === '1' ? 'selected' : '' ?>>Active Only</option>
                    <option value="0" <?= ($_GET['is_active'] ?? '') === '0' ? 'selected' : '' ?>>Inactive Only</option>
                </select>
                
                <input type="text" name="search" placeholder="Search by title..." 
                       value="<?= e($_GET['search'] ?? '') ?>" style="flex: 1; min-width: 200px;">
                
                <button type="submit" class="btn">Filter</button>
                <?php if (!empty($_GET)): ?>
                    <a href="<?= base_url('admin/eye-exercises.php') ?>" class="btn" style="background: #6c757d;">Clear</a>
                <?php endif; ?>
            </form>
        </div>
        
        <table>
            <thead>
                <tr>
                    <th>ID</th>
                    <th>Title</th>
                    <th>Slug</th>
                    <th>Category</th>
                    <th>Difficulty</th>
                    <th>Duration</th>
                    <th>Active</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php if (empty($exercises)): ?>
                    <tr>
                        <td colspan="8" style="text-align: center; padding: 40px;">
                            No exercises found. <a href="<?= base_url('admin/eye-exercises-edit.php') ?>">Create your first exercise</a>
                        </td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($exercises as $exercise): ?>
                    <tr>
                        <td><?= e($exercise['id']) ?></td>
                        <td>
                            <strong><?= e($exercise['title']) ?></strong>
                            <?php if (!empty($exercise['short_description'])): ?>
                                <br><small style="color: #666;"><?= e(substr($exercise['short_description'], 0, 60)) ?><?= strlen($exercise['short_description']) > 60 ? '...' : '' ?></small>
                            <?php endif; ?>
                        </td>
                        <td><code><?= e($exercise['slug']) ?></code></td>
                        <td><span class="badge badge-category"><?= e($exercise['category']) ?></span></td>
                        <td><span class="badge badge-<?= e($exercise['difficulty']) ?>"><?= e($exercise['difficulty']) ?></span></td>
                        <td>
                            <?php
                            $mins = floor($exercise['default_duration_seconds'] / 60);
                            $secs = $exercise['default_duration_seconds'] % 60;
                            echo $mins > 0 ? "{$mins}m " : '';
                            echo "{$secs}s";
                            ?>
                        </td>
                        <td>
                            <form method="POST" style="display: inline;">
                                <?= csrf_field() ?>
                                <input type="hidden" name="action" value="toggle_active">
                                <input type="hidden" name="id" value="<?= e($exercise['id']) ?>">
                                <label class="toggle-switch">
                                    <input type="checkbox" <?= $exercise['is_active'] ? 'checked' : '' ?> onchange="this.form.submit()">
                                    <span class="slider"></span>
                                </label>
                            </form>
                        </td>
                        <td>
                            <a href="<?= base_url('admin/eye-exercises-edit.php?id=' . $exercise['id']) ?>" class="btn btn-sm">Edit</a>
                            <form method="POST" style="display: inline;" onsubmit="return confirm('Delete this exercise? This cannot be undone.');">
                                <?= csrf_field() ?>
                                <input type="hidden" name="action" value="delete">
                                <input type="hidden" name="id" value="<?= e($exercise['id']) ?>">
                                <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </tbody>
        </table>
    </div>
</body>
</html>
