<?php
require_once __DIR__ . '/../../includes/bootstrap.php';

requireAdmin();

$routineController = new EyeRoutineController(getPDO());
$exerciseController = new EyeExerciseController(getPDO());

$editMode = false;
$routine = null;
$assignedExercises = [];

// Load routine for editing
if (isset($_GET['id'])) {
    $editMode = true;
    $routine = $routineController->getRoutineById($_GET['id'], true);
    if (!$routine) {
        flash('flash_message', 'Routine not found.', 'danger');
        redirect(base_url('admin/eye-routines.php'));
    }
    $assignedExercises = $routine['exercises'] ?? [];
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    verify_csrf();
    
    // Build routine metadata
    $data = [
        'title' => $_POST['title'] ?? '',
        'slug' => $_POST['slug'] ?? '',
        'description' => $_POST['description'] ?? '',
        'difficulty' => $_POST['difficulty'] ?? 'easy',
        'is_ai_generated' => isset($_POST['is_ai_generated']) ? 1 : 0,
        'is_active' => isset($_POST['is_active']) ? 1 : 0
    ];
    
    // Create or update routine
    if ($editMode) {
        $routineId = $_POST['routine_id'] ?? null;
        if (!$routineController->updateRoutine($routineId, $data)) {
            flash('flash_message', 'Error updating routine. Check all required fields.', 'danger');
            redirect(base_url('admin/eye-routines-edit.php?id=' . $routineId));
        }
    } else {
        $routineId = $routineController->createRoutine($data);
        if (!$routineId) {
            flash('flash_message', 'Error creating routine. Check all required fields.', 'danger');
            redirect(base_url('admin/eye-routines-edit.php'));
        }
    }
    
    // Handle exercise assignments
    $exercisesData = [];
    if (isset($_POST['assigned_exercises']) && !empty($_POST['assigned_exercises'])) {
        $assignedIds = json_decode($_POST['assigned_exercises'], true);
        $sortOrder = 1;
        
        foreach ($assignedIds as $exerciseId) {
            $exercisesData[] = [
                'exercise_id' => $exerciseId,
                'sort_order' => $sortOrder++,
                'repetitions' => (int)($_POST['repetitions_' . $exerciseId] ?? 1),
                'override_duration_seconds' => !empty($_POST['override_duration_' . $exerciseId]) 
                    ? (int)$_POST['override_duration_' . $exerciseId] 
                    : null
            ];
        }
    }
    
    // Attach exercises to routine
    if ($routineController->attachExercisesToRoutine($routineId, $exercisesData)) {
        flash('flash_message', ($editMode ? 'Routine updated' : 'Routine created') . ' successfully!', 'success');
        redirect(base_url('admin/eye-routines.php'));
    } else {
        flash('flash_message', 'Error attaching exercises to routine.', 'danger');
    }
}

// Get all active exercises for selection
$allExercises = $exerciseController->getAllExercises(['is_active' => 1]);

$pageTitle = ($editMode ? 'Edit' : 'Create New') . ' Eye Routine - Admin';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= e($pageTitle) ?></title>
    <link rel="stylesheet" href="<?= base_url('assets/css/style.css') ?>">
    <style>
        .admin-container { max-width: 1400px; margin: 40px auto; padding: 20px; }
        .admin-nav { background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 30px; }
        .admin-nav a { margin-right: 15px; padding: 8px 15px; background: #007bff; color: white; text-decoration: none; border-radius: 5px; display: inline-block; margin-bottom: 5px; }
        .admin-nav a:hover { background: #0056b3; }
        .form-group { margin-bottom: 20px; }
        .form-group label { display: block; font-weight: bold; margin-bottom: 5px; }
        .form-group input, .form-group textarea, .form-group select { width: 100%; padding: 10px; border: 1px solid #ddd; border-radius: 5px; font-size: 14px; font-family: inherit; }
        .form-group textarea { min-height: 80px; resize: vertical; }
        .form-group small { display: block; color: #666; margin-top: 5px; font-size: 13px; }
        .checkbox-group label { display: inline-flex; align-items: center; font-weight: normal; margin-right: 20px; }
        .checkbox-group input { width: auto; margin-right: 8px; }
        .btn { padding: 12px 24px; background: #007bff; color: white; border: none; border-radius: 5px; cursor: pointer; font-size: 16px; text-decoration: none; display: inline-block; }
        .btn:hover { background: #0056b3; }
        .btn-secondary { background: #6c757d; }
        .btn-secondary:hover { background: #5a6268; }
        .btn-danger { background: #dc3545; }
        .btn-danger:hover { background: #c82333; }
        .btn-sm { padding: 6px 12px; font-size: 13px; }
        .alert { padding: 15px; border-radius: 5px; margin-bottom: 20px; }
        .alert-success { background: #d4edda; color: #155724; }
        .alert-danger { background: #f8d7da; color: #721c24; }
        .alert-info { background: #d1ecf1; color: #0c5460; }
        
        /* Exercise Assignment UI */
        .assignment-container { display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 30px 0; }
        .exercises-panel { border: 2px solid #ddd; border-radius: 8px; padding: 20px; background: #f8f9fa; }
        .exercises-panel h3 { margin-top: 0; color: #333; }
        .exercise-search { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 5px; margin-bottom: 15px; }
        .exercise-item { background: white; border: 1px solid #ddd; border-radius: 5px; padding: 10px; margin-bottom: 10px; cursor: pointer; transition: all 0.2s; }
        .exercise-item:hover { border-color: #007bff; box-shadow: 0 2px 5px rgba(0,123,255,0.2); }
        .exercise-item.selected { background: #e7f3ff; border-color: #007bff; }
        .exercise-item strong { display: block; color: #333; margin-bottom: 5px; }
        .exercise-item small { color: #666; font-size: 12px; }
        .badge { padding: 3px 6px; border-radius: 3px; font-size: 10px; font-weight: bold; text-transform: uppercase; margin-right: 5px; }
        .badge-easy { background: #28a745; color: white; }
        .badge-medium { background: #ffc107; color: #333; }
        .badge-hard { background: #dc3545; color: white; }
        
        .assigned-list { list-style: none; padding: 0; margin: 0; min-height: 200px; }
        .assigned-item { background: white; border: 1px solid #ddd; border-radius: 5px; padding: 15px; margin-bottom: 10px; }
        .assigned-item-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 10px; }
        .assigned-item-header strong { color: #007bff; }
        .assigned-item-controls { display: grid; grid-template-columns: 1fr 1fr; gap: 10px; }
        .assigned-item-controls label { font-size: 13px; font-weight: bold; }
        .assigned-item-controls input { width: 100%; padding: 6px; border: 1px solid #ddd; border-radius: 4px; }
        .move-buttons { display: flex; gap: 5px; }
        .move-buttons button { padding: 4px 8px; font-size: 12px; background: #6c757d; border: none; color: white; border-radius: 3px; cursor: pointer; }
        .move-buttons button:hover { background: #5a6268; }
        .empty-state { text-align: center; padding: 40px; color: #999; border: 2px dashed #ddd; border-radius: 5px; }
        .total-duration { background: #007bff; color: white; padding: 15px; border-radius: 5px; text-align: center; font-size: 18px; font-weight: bold; margin-top: 15px; }
    </style>
</head>
<body>
    <div class="admin-container">
        <h1><?= $editMode ? 'Edit' : 'Create New' ?> Eye Routine</h1>
        
        <?php include 'partials/admin-nav.php'; ?>
        
        <?php flash('flash_message'); ?>
        
        <form method="POST" id="routineForm">
            <?= csrf_field() ?>
            <?php if ($editMode): ?>
                <input type="hidden" name="routine_id" value="<?= e($routine['id']) ?>">
            <?php endif; ?>
            
            <div class="form-group">
                <label for="title">Title <span style="color: #dc3545;">*</span></label>
                <input type="text" id="title" name="title" required
                       value="<?= $editMode ? e($routine['title']) : '' ?>">
            </div>
            
            <div class="form-group">
                <label for="slug">Slug <span style="color: #dc3545;">*</span></label>
                <input type="text" id="slug" name="slug"
                       value="<?= $editMode ? e($routine['slug']) : '' ?>">
                <small>Auto-generated from title if left empty</small>
            </div>
            
            <div class="form-group">
                <label for="description">Description</label>
                <textarea id="description" name="description" rows="3"><?= $editMode ? e($routine['description']) : '' ?></textarea>
            </div>
            
            <div class="form-group">
                <label for="difficulty">Difficulty <span style="color: #dc3545;">*</span></label>
                <select id="difficulty" name="difficulty" required>
                    <option value="easy" <?= ($editMode && $routine['difficulty'] === 'easy') ? 'selected' : '' ?>>Easy</option>
                    <option value="medium" <?= ($editMode && $routine['difficulty'] === 'medium') ? 'selected' : '' ?>>Medium</option>
                    <option value="hard" <?= ($editMode && $routine['difficulty'] === 'hard') ? 'selected' : '' ?>>Hard</option>
                </select>
            </div>
            
            <div class="checkbox-group">
                <label>
                    <input type="checkbox" name="is_active" value="1" 
                           <?= ($editMode && $routine['is_active']) || !$editMode ? 'checked' : '' ?>>
                    Active (visible to users)
                </label>
                <label>
                    <input type="checkbox" name="is_ai_generated" value="1"
                           <?= ($editMode && $routine['is_ai_generated']) ? 'checked' : '' ?>>
                    AI Generated
                </label>
            </div>
            
            <hr style="margin: 30px 0; border: none; border-top: 2px solid #ddd;">
            
            <h2>Assign Exercises to Routine</h2>
            <div class="alert alert-info">
                Click exercises on the left to add them to the routine on the right. Set repetitions and override duration for each. Use arrows to reorder.
            </div>
            
            <div class="assignment-container">
                <!-- Available Exercises -->
                <div class="exercises-panel">
                    <h3>Available Exercises</h3>
                    <input type="text" class="exercise-search" id="exerciseSearch" placeholder="Search exercises...">
                    <div id="availableExercises">
                        <?php foreach ($allExercises as $exercise): ?>
                        <div class="exercise-item" data-id="<?= e($exercise['id']) ?>" 
                             data-title="<?= e($exercise['title']) ?>"
                             data-category="<?= e($exercise['category']) ?>"
                             data-difficulty="<?= e($exercise['difficulty']) ?>"
                             data-duration="<?= e($exercise['default_duration_seconds']) ?>"
                             onclick="addExercise(<?= e($exercise['id']) ?>)">
                            <strong><?= e($exercise['title']) ?></strong>
                            <small>
                                <span class="badge badge-<?= e($exercise['difficulty']) ?>"><?= e($exercise['difficulty']) ?></span>
                                <?= e($exercise['category']) ?> • 
                                <?php
                                $mins = floor($exercise['default_duration_seconds'] / 60);
                                $secs = $exercise['default_duration_seconds'] % 60;
                                echo ($mins > 0 ? "{$mins}m " : '') . "{$secs}s";
                                ?>
                            </small>
                        </div>
                        <?php endforeach; ?>
                    </div>
                </div>
                
                <!-- Assigned Exercises -->
                <div class="exercises-panel">
                    <h3>Assigned Exercises</h3>
                    <ul class="assigned-list" id="assignedExercises">
                        <?php if (empty($assignedExercises)): ?>
                        <div class="empty-state" id="emptyState">
                            No exercises assigned yet.<br>Click exercises on the left to add them.
                        </div>
                        <?php else: ?>
                        <?php foreach ($assignedExercises as $index => $ex): ?>
                        <li class="assigned-item" data-id="<?= e($ex['exercise_id']) ?>" 
                            data-duration="<?= e($ex['exercise_default_duration']) ?>">
                            <div class="assigned-item-header">
                                <strong><?= ($index + 1) ?>. <?= e($ex['exercise_title']) ?></strong>
                                <div class="move-buttons">
                                    <button type="button" onclick="moveUp(this)" <?= $index === 0 ? 'disabled' : '' ?>>▲</button>
                                    <button type="button" onclick="moveDown(this)" <?= $index === count($assignedExercises) - 1 ? 'disabled' : '' ?>>▼</button>
                                    <button type="button" onclick="removeExercise(this)" style="background: #dc3545;">✕</button>
                                </div>
                            </div>
                            <div class="assigned-item-controls">
                                <div>
                                    <label>Repetitions:</label>
                                    <input type="number" min="1" max="20" value="<?= e($ex['repetitions']) ?>" 
                                           class="reps-input" onchange="updateDuration()">
                                </div>
                                <div>
                                    <label>Override Duration (seconds):</label>
                                    <input type="number" min="5" max="3600" 
                                           value="<?= $ex['override_duration_seconds'] ?? '' ?>" 
                                           class="duration-input" placeholder="<?= e($ex['exercise_default_duration']) ?>" 
                                           onchange="updateDuration()">
                                </div>
                            </div>
                            <small style="color: #666; margin-top: 5px; display: block;">
                                Default: <?= e($ex['exercise_default_duration']) ?>s | 
                                Category: <?= e($ex['exercise_category']) ?>
                            </small>
                        </li>
                        <?php endforeach; ?>
                        <?php endif; ?>
                    </ul>
                    <div class="total-duration" id="totalDuration">
                        Total Duration: 0m 0s
                    </div>
                </div>
            </div>
            
            <input type="hidden" name="assigned_exercises" id="assignedExercisesInput">
            
            <div style="margin-top: 30px; padding-top: 20px; border-top: 2px solid #ddd;">
                <button type="submit" class="btn"><?= $editMode ? 'Update' : 'Create' ?> Routine</button>
                <a href="<?= base_url('admin/eye-routines.php') ?>" class="btn btn-secondary">Cancel</a>
                
                <?php if ($editMode): ?>
                    <form method="POST" action="<?= base_url('admin/eye-routines.php') ?>" style="display: inline; float: right;"
                          onsubmit="return confirm('Delete this routine? This cannot be undone.');">
                        <?= csrf_field() ?>
                        <input type="hidden" name="action" value="delete">
                        <input type="hidden" name="id" value="<?= e($routine['id']) ?>">
                        <button type="submit" class="btn btn-danger">Delete Routine</button>
                    </form>
                <?php endif; ?>
            </div>
        </form>
    </div>
    
    <script>
        // Auto-generate slug from title
        document.getElementById('title').addEventListener('input', function(e) {
            const slugField = document.getElementById('slug');
            if (!slugField.dataset.manuallyEdited) {
                const slug = e.target.value.toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-+|-+$/g, '');
                slugField.value = slug;
            }
        });
        
        document.getElementById('slug').addEventListener('input', function() {
            this.dataset.manuallyEdited = 'true';
        });
        
        // Exercise search filter
        document.getElementById('exerciseSearch').addEventListener('input', function(e) {
            const query = e.target.value.toLowerCase();
            document.querySelectorAll('#availableExercises .exercise-item').forEach(item => {
                const title = item.dataset.title.toLowerCase();
                const category = item.dataset.category.toLowerCase();
                item.style.display = (title.includes(query) || category.includes(query)) ? 'block' : 'none';
            });
        });
        
        // Add exercise to routine
        function addExercise(exerciseId) {
            const exerciseEl = document.querySelector(`.exercise-item[data-id="${exerciseId}"]`);
            
            // Check if already added
            if (document.querySelector(`#assignedExercises .assigned-item[data-id="${exerciseId}"]`)) {
                alert('This exercise is already in the routine!');
                return;
            }
            
            // Remove empty state
            const emptyState = document.getElementById('emptyState');
            if (emptyState) emptyState.remove();
            
            // Create assigned item
            const assignedList = document.getElementById('assignedExercises');
            const li = document.createElement('li');
            li.className = 'assigned-item';
            li.dataset.id = exerciseId;
            li.dataset.duration = exerciseEl.dataset.duration;
            
            const currentCount = assignedList.querySelectorAll('.assigned-item').length;
            
            li.innerHTML = `
                <div class="assigned-item-header">
                    <strong>${currentCount + 1}. ${exerciseEl.dataset.title}</strong>
                    <div class="move-buttons">
                        <button type="button" onclick="moveUp(this)">▲</button>
                        <button type="button" onclick="moveDown(this)">▼</button>
                        <button type="button" onclick="removeExercise(this)" style="background: #dc3545;">✕</button>
                    </div>
                </div>
                <div class="assigned-item-controls">
                    <div>
                        <label>Repetitions:</label>
                        <input type="number" min="1" max="20" value="1" class="reps-input" onchange="updateDuration()">
                    </div>
                    <div>
                        <label>Override Duration (seconds):</label>
                        <input type="number" min="5" max="3600" value="" class="duration-input" 
                               placeholder="${exerciseEl.dataset.duration}" onchange="updateDuration()">
                    </div>
                </div>
                <small style="color: #666; margin-top: 5px; display: block;">
                    Default: ${exerciseEl.dataset.duration}s | Category: ${exerciseEl.dataset.category}
                </small>
            `;
            
            assignedList.appendChild(li);
            exerciseEl.classList.add('selected');
            updateDuration();
            updateMoveButtons();
        }
        
        // Remove exercise
        function removeExercise(btn) {
            const item = btn.closest('.assigned-item');
            const exerciseId = item.dataset.id;
            
            // Unmark in available list
            const availableItem = document.querySelector(`#availableExercises .exercise-item[data-id="${exerciseId}"]`);
            if (availableItem) availableItem.classList.remove('selected');
            
            item.remove();
            renumberExercises();
            updateDuration();
            updateMoveButtons();
            
            // Show empty state if no exercises
            if (document.querySelectorAll('#assignedExercises .assigned-item').length === 0) {
                document.getElementById('assignedExercises').innerHTML = '<div class="empty-state" id="emptyState">No exercises assigned yet.<br>Click exercises on the left to add them.</div>';
            }
        }
        
        // Move exercise up
        function moveUp(btn) {
            const item = btn.closest('.assigned-item');
            const prev = item.previousElementSibling;
            if (prev && prev.classList.contains('assigned-item')) {
                item.parentNode.insertBefore(item, prev);
                renumberExercises();
                updateMoveButtons();
            }
        }
        
        // Move exercise down
        function moveDown(btn) {
            const item = btn.closest('.assigned-item');
            const next = item.nextElementSibling;
            if (next && next.classList.contains('assigned-item')) {
                item.parentNode.insertBefore(next, item);
                renumberExercises();
                updateMoveButtons();
            }
        }
        
        // Renumber exercises
        function renumberExercises() {
            document.querySelectorAll('#assignedExercises .assigned-item').forEach((item, index) => {
                const header = item.querySelector('.assigned-item-header strong');
                const title = header.textContent.replace(/^\d+\.\s*/, '');
                header.textContent = `${index + 1}. ${title}`;
            });
        }
        
        // Update move button states
        function updateMoveButtons() {
            const items = document.querySelectorAll('#assignedExercises .assigned-item');
            items.forEach((item, index) => {
                const upBtn = item.querySelector('.move-buttons button:nth-child(1)');
                const downBtn = item.querySelector('.move-buttons button:nth-child(2)');
                upBtn.disabled = index === 0;
                downBtn.disabled = index === items.length - 1;
            });
        }
        
        // Calculate and update total duration
        function updateDuration() {
            let total = 0;
            document.querySelectorAll('#assignedExercises .assigned-item').forEach(item => {
                const reps = parseInt(item.querySelector('.reps-input').value) || 1;
                const override = item.querySelector('.duration-input').value;
                const defaultDur = parseInt(item.dataset.duration);
                const duration = override ? parseInt(override) : defaultDur;
                total += reps * duration;
            });
            
            const mins = Math.floor(total / 60);
            const secs = total % 60;
            document.getElementById('totalDuration').textContent = `Total Duration: ${mins}m ${secs}s`;
        }
        
        // Serialize assigned exercises before form submit
        document.getElementById('routineForm').addEventListener('submit', function(e) {
            const assignedIds = [];
            const items = document.querySelectorAll('#assignedExercises .assigned-item');
            
            if (items.length === 0) {
                alert('Please add at least one exercise to the routine!');
                e.preventDefault();
                return false;
            }
            
            items.forEach(item => {
                const id = item.dataset.id;
                assignedIds.push(id);
                
                // Set hidden inputs for repetitions and override duration
                const reps = item.querySelector('.reps-input').value;
                const override = item.querySelector('.duration-input').value;
                
                // Create hidden inputs
                let repsInput = document.querySelector(`input[name="repetitions_${id}"]`);
                if (!repsInput) {
                    repsInput = document.createElement('input');
                    repsInput.type = 'hidden';
                    repsInput.name = `repetitions_${id}`;
                    this.appendChild(repsInput);
                }
                repsInput.value = reps;
                
                if (override) {
                    let overrideInput = document.querySelector(`input[name="override_duration_${id}"]`);
                    if (!overrideInput) {
                        overrideInput = document.createElement('input');
                        overrideInput.type = 'hidden';
                        overrideInput.name = `override_duration_${id}`;
                        this.appendChild(overrideInput);
                    }
                    overrideInput.value = override;
                }
            });
            
            document.getElementById('assignedExercisesInput').value = JSON.stringify(assignedIds);
        });
        
        // Initialize
        updateDuration();
        updateMoveButtons();
    </script>
</body>
</html>
